<?php
session_start();
include 'db.php';

// Aggiungi utente nuovo
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_user'])) {
    $name = trim($_POST['user_name']);
    $email = trim($_POST['user_email']);
    if ($name !== '' && filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $stmt = $conn->prepare("INSERT INTO users (name, email) VALUES (?, ?)");
        $stmt->bind_param("ss", $name, $email);
        $stmt->execute();
        $stmt->close();
        $_SESSION['message'] = "Utente aggiunto con successo!";
    } else {
        $_SESSION['message'] = "Nome o email non validi.";
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Per pre-selezione utente da GET (opzionale)
$preselectedUserId = isset($_GET['user']) ? intval($_GET['user']) : 0;

// Messaggio di feedback
$message = $_SESSION['message'] ?? '';
unset($_SESSION['message']);
?>

<!DOCTYPE html>
<html lang="it">
<head>
  <meta charset="UTF-8" />
  <title>Monitor seriale UID e Noleggi</title>
  <style>
    body { font-family: Arial; padding: 20px; }
    h2 { margin-top: 30px; }
    textarea { width: 100%; height: 120px; }
    ul { background: #eee; padding: 10px; list-style-type:none; }
    ul li { margin-bottom: 4px; }
    .ok { color: green; }
    .not-found { color: red; }
    #noleggia { margin-top: 20px; padding: 10px 20px; }
    label, select {
      display: block;
      margin-top: 20px;
      font-weight: bold;
    }
    .message { margin-bottom: 20px; color: blue; }
    form#addUserForm { margin-bottom: 40px; }
    form#addUserForm input { padding: 6px; margin-right: 10px; }
    form#addUserForm button { padding: 6px 14px; }
  </style>
</head>
<body>
  <h1>Monitor seriale UID e Noleggi</h1>

  <?php if ($message): ?>
    <div class="message"><?= htmlspecialchars($message) ?></div>
  <?php endif; ?>

  <h3>Aggiungi nuovo utente (opzionale)</h3>
  <form id="addUserForm" method="POST" action="<?= htmlspecialchars($_SERVER['PHP_SELF']) ?>">
    <input type="hidden" name="add_user" value="1" />
    <input type="text" name="user_name" placeholder="Nome utente" required>
    <input type="email" name="user_email" placeholder="Email utente" required>
    <button type="submit">Aggiungi Utente</button>
  </form>

  <button id="connect">Collega seriale</button>

  <h2>Log ricevuto</h2>
  <textarea id="log" readonly></textarea>

  <h2>Dispositivi aggiunti</h2>
  <ul id="devices"></ul>

  <form id="rentalForm" action="noleggi.php" method="GET" onsubmit="return prepareSubmit()">
    <label for="userSelect">Seleziona utente</label>
    <select id="userSelect" name="user" required>
      <option value="">-- Seleziona utente --</option>
      <?php
      $res = $conn->query("SELECT id, name FROM users ORDER BY name");
      while ($row = $res->fetch_assoc()) {
        $selected = ($preselectedUserId === intval($row['id'])) ? 'selected' : '';
        echo "<option value=\"" . intval($row['id']) . "\" $selected>" . htmlspecialchars($row['name']) . "</option>";
      }
      ?>
    </select>

    <input type="hidden" name="comment" id="commentField" value="">

    <button type="submit" id="noleggia">Noleggia</button>
  </form>

  <script>
    let port, reader;
    const logArea = document.getElementById("log");
    const deviceList = document.getElementById("devices");
    const commentField = document.getElementById("commentField");
    const userSelect = document.getElementById("userSelect");

    let rawBuffer = "";
    const addedDevices = new Set();

    async function connectSerial() {
      try {
        port = await navigator.serial.requestPort();
        await port.open({ baudRate: 115200 });

        reader = port.readable.getReader();
        readLoop();
      } catch (e) {
        alert("Errore: " + e);
      }
    }

    async function readLoop() {
      while (true) {
        const { value, done } = await reader.read();
        if (done) break;
        const chunk = new TextDecoder().decode(value);

        logArea.value += chunk;
        logArea.scrollTop = logArea.scrollHeight;

        rawBuffer += chunk.replace(/\s/g, '');
        const matches = rawBuffer.match(/[0-9A-F]{2}(:[0-9A-F]{2}){6}/i);
        if (matches) {
          const uid = matches[0].toUpperCase();
          rawBuffer = "";
          checkUID(uid);
        }
      }
    }

    async function checkUID(uid) {
      try {
        const res = await fetch("check_uid.php?uid=" + uid);
        const data = await res.json();

        if (data.success && !addedDevices.has(data.device)) {
          addedDevices.add(data.device);
          const li = document.createElement("li");
          li.textContent = data.device;
          deviceList.prepend(li);
          updateCommentField();
        } else if (!data.success) {
          const li = document.createElement("li");
          li.innerHTML = `<span class="not-found">❌ ${uid} → Non trovato</span>`;
          deviceList.prepend(li);
        }
      } catch (err) {
        console.error(err);
      }
    }

    function updateCommentField() {
      commentField.value = Array.from(addedDevices).join(">");
    }

    function prepareSubmit() {
      if (userSelect.value === "") {
        alert("Seleziona un utente prima di noleggiare.");
        return false;
      }
      updateCommentField();
      if (commentField.value === "") {
        alert("Non ci sono dispositivi da noleggiare.");
        return false;
      }
      return true; // Invia form con user=...&comment=...
    }

    document.getElementById("connect").addEventListener("click", connectSerial);
  </script>
</body>
</html>
