<?php
include 'db.php';
require 'vendor/autoload.php'; // PHPMailer
require('fpdf/fpdf.php');

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

session_start();

if (!isset($_SESSION['step'])) $_SESSION['step'] = 1;

// Aggiungi utente nuovo
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_user'])) {
    $name = trim($_POST['user_name']);
    $email = trim($_POST['user_email']);
    if ($name !== '' && filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $stmt = $conn->prepare("INSERT INTO users (name, email) VALUES (?, ?)");
        $stmt->bind_param("ss", $name, $email);
        $stmt->execute();
        $stmt->close();
        $_SESSION['message'] = "Utente aggiunto con successo!";
    } else {
        $_SESSION['message'] = "Nome o email non validi.";
    }
    header("Location: noleggi.php");
    exit();
}

// Step 1: selezione utente e dispositivi
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['step1_submit'])) {
    $user_id = intval($_POST['user_id'] ?? 0);
    $device_codes = $_POST['device_codes'] ?? [];
    if ($user_id > 0 && count($device_codes) > 0) {
        $_SESSION['noleggio_user_id'] = $user_id;
        $_SESSION['noleggio_devices'] = $device_codes;
        $_SESSION['step'] = 2;
    } else {
        $_SESSION['message'] = "Seleziona utente e almeno un dispositivo.";
    }
    header("Location: noleggi.php");
    exit();
}

// Step 2: firma e conferma noleggio
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['step2_submit'])) {
    $signatureData = $_POST['signature'] ?? '';
    if (empty($signatureData)) {
        $_SESSION['message'] = "Firma obbligatoria per completare il noleggio.";
        header("Location: noleggi.php");
        exit();
    }

    $user_id = $_SESSION['noleggio_user_id'] ?? 0;
    $device_codes = $_SESSION['noleggio_devices'] ?? [];

    if ($user_id == 0 || count($device_codes) == 0) {
        $_SESSION['message'] = "Dati noleggio mancanti.";
        $_SESSION['step'] = 1;
        header("Location: noleggi.php");
        exit();
    }

    $rental_start = date('Y-m-d H:i:s');

    // Recupera dati utente
    $userQ = $conn->prepare("SELECT name, email FROM users WHERE id = ?");
    $userQ->bind_param("i", $user_id);
    $userQ->execute();
    $userQ->bind_result($user_name, $user_email);
    $userQ->fetch();
    $userQ->close();

    // Inserisci primo noleggio per recuperare ID
    $stmt = $conn->prepare("INSERT INTO rentals (device_code, user_id, rental_start) VALUES (?, ?, ?)");
    $stmt->bind_param("sis", $device_codes[0], $user_id, $rental_start);
    $stmt->execute();
    $rental_id = $stmt->insert_id;

    // Inserisci altri noleggi se presenti
    for ($i = 1; $i < count($device_codes); $i++) {
        $code = $device_codes[$i];
        $stmt->bind_param("sis", $code, $user_id, $rental_start);
        $stmt->execute();
    }
    $stmt->close();

    // Salva firma come PNG
    $signatureData = str_replace('data:image/png;base64,', '', $signatureData);
    $signatureData = str_replace(' ', '+', $signatureData);
    $signatureBin = base64_decode($signatureData);

    if (!is_dir('noleggi_pdf')) {
        mkdir('noleggi_pdf', 0777, true);
    }
    $signaturePath = "noleggi_pdf/signature_{$rental_id}.png";
    file_put_contents($signaturePath, $signatureBin);

    // Genera PDF
class PDF extends FPDF {
    function Header() {
        if (file_exists('fermi.png')) {
            $this->Image('fermi.png', 10, 10, 190);
            $this->Ln(30);
        }
        $this->SetFont('Arial','B',12);
        $this->Ln(15);
    }
    function Footer() {
        $this->SetY(-15);
        $this->SetFont('Arial','I',8);
        $this->Cell(0,10,'Pagina '.$this->PageNo().'/{nb}',0,0,'C');
        $this->SetY(-10);
        $this->SetFont('Arial','I',5);
        $this->Cell(0,10,'Realizzato da RICCHIUTO Michele 4CI. Tutti i diritti riservati',0,0,'C');
    }
}

$pdf = new PDF();
$pdf->AliasNbPages();
$pdf->AddPage();
$pdf->SetFont('Arial','',12);

$pdf->Cell(0,10, "Noleggio ID: $rental_id", 0,1);
$pdf->Cell(0,10, "Nome utente: $user_name", 0,1);
$pdf->Cell(0,10, "Email: $user_email", 0,1);
$pdf->Cell(0,10, "Data inizio noleggio: $rental_start", 0,1);
$pdf->Ln(5);

// Tabella dispositivi
$pdf->SetFont('Arial','B',12);
$pdf->Cell(60,10,'Codice',1);
$pdf->Cell(60,10,'Inventario',1);
$pdf->Ln();

$pdf->SetFont('Arial','',12);
foreach ($device_codes as $code) {
    $deviceQ = $conn->prepare("SELECT numero_inventario FROM device WHERE codice = ?");
    $deviceQ->bind_param("s", $code);
    $deviceQ->execute();
    $deviceQ->bind_result($inv_num);
    $deviceQ->fetch();
    $deviceQ->close();

    $pdf->Cell(60,10,$code,1);
    $pdf->Cell(60,10,$inv_num,1);
    $pdf->Ln();
}
$pdf->Ln(10);

// ✅ Carico di responsabilità
$pdf->SetFont('Arial', '', 11);
$pdf->SetTextColor(100, 100, 100); // Grigio
$pdf->MultiCell(0, 8, "Con la presente, l'utente si assume ogni responsabilita' relativa all'utilizzo, custodia e restituzione dell'attrezzatura noleggiata. In caso di smarrimento o danneggiamento, si impegna a comunicarlo tempestivamente e a rispondere di eventuali costi di riparazione o sostituzione. \n E' severamente vietato portare al di fuori dell' Istituto qualsiasi dispositivo noleggiato. L' attrezzatura deve essere utilizzata esclusivamente all' interno dei locali scolastici.");

// Spazio prima firme
$pdf->Ln(15);

// Titoli firme
$pdf->SetFont('Arial', 'B', 12);
$pdf->SetTextColor(0, 0, 0);
$pdf->Cell(90, 10, "Firma Cliente", 0, 0, 'L');
$pdf->SetX(155); // Sposta più a destra il testo "Firma Ufficio"
$pdf->Cell(40, 10, "Firma Ufficio", 0, 1, 'L');

// Firma cliente e firma ufficio allineate
$y = $pdf->GetY();
if (file_exists($signaturePath)) {
    $pdf->Image($signaturePath, 10, $y, 60); // Firma Cliente (es. 60x30)
}
if (file_exists('firma_uff.png')) {
    $pdf->Image('firma_uff.png', 155, $y, 30); // Firma Ufficio 30x30, 1:1
}


$pdf->Ln(35); // Spazio dopo le firme

// Salvataggio PDF
$safe_name = strtolower(preg_replace("/[^a-zA-Z0-9]+/", "_", "$user_email"));
$safe_nome = strtolower(preg_replace("/[^a-zA-Z0-9]+/", "_", "$user_name"));
$pdfFilename = "noleggio_{$safe_name}_{$safe_nome}_{$rental_id}.pdf";
$pdfPath = "noleggi_pdf/$pdfFilename";
$pdf->Output('F', $pdfPath);


    // Invia email all'utente
    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = 'smtp.gmail.com';
        $mail->SMTPAuth = true;
        $mail->Username = 'ricchiutomichele8@gmail.com';
        $mail->Password = 'pdnt jsyi wipd trxb';
        $mail->SMTPSecure = 'tls';
        $mail->Port = 587;

        $mail->setFrom('LA_TUA_EMAIL@gmail.com', 'Laboratorio Classroom');
        $mail->addAddress($user_email);
        $mail->addCC('visori@fermilecce.edu.it');
        $mail->Subject = 'Riepilogo Noleggio Dispositivo';
        $mail->Body = "Gentile $user_name,\n\nIn allegato trova il riepilogo del noleggio da Lei eseguito.\n\nID: $rental_id \n\nCordiali saluti \nLaboratorio Classroom.";
        $mail->addAttachment($pdfPath);
        $mail->send();
    } catch (Exception $e) {
        $_SESSION['message'] .= "<br>Errore invio email: {$mail->ErrorInfo}";
    }

    unset($_SESSION['noleggio_user_id'], $_SESSION['noleggio_devices']);
    $_SESSION['step'] = 1;

    $_SESSION['message'] = "Noleggio completato! PDF salvato: <a href='$pdfPath' target='_blank'>Scarica PDF</a>";
    header("Location: noleggi.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="it">
<head>
    <meta charset="UTF-8" />
    <title>Gestione Noleggi - Nuovo noleggio</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        body { background:#f9f9f9; font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; }
        .container { max-width: 900px; margin: 40px auto; }
        canvas { border: 1px solid #ccc; background: white; border-radius: 8px; }
        .signature-container { margin-top: 15px; }
        .btn-clear { margin-left: 10px; }
        .message { margin-bottom: 20px; }
        /* Navbar - forza sempre il menu in orizzontale */
        .navbar-nav {
            flex-direction: row !important;  /* riga sempre */
            flex-wrap: nowrap !important;    /* NO ritorno a capo */
            overflow-x: auto;                /* scroll orizzontale se troppo lungo */
            -webkit-overflow-scrolling: touch; /* smooth scroll su mobile */

        }
        .navbar-brand, .nav-link {
            color: black;
            font-weight: 600;
            cursor: pointer;

        }
        .nav-link:hover, .nav-link:focus {
            text-decoration: underline;
            color: #350050ff !important;
        }
        /* Spaziatura tra i link */
        .nav-item {
            margin-right: 1rem;
        }
    </style>
</head>
<body>

<nav class="navbar navbar-expand-lg navbar-dark">
    <div class="container-fluid">
        <a class="navbar-brand" href="#">Gestione Noleggi</a>
        <ul class="navbar-nav ms-auto mb-2 mb-lg-0">
            <li class="nav-item"><a class="nav-link" onclick="location.href='index.php'">Indietro</a></li>
            <!-- <li class="nav-item"><a class="nav-link" onclick="showSection('nuovo_noleggio')">Nuovo Noleggio</a></li> -->
        </ul>
    </div>
</nav>
<div class="container">
    <h2 class="mb-4">Gestione Noleggi - Nuovo Noleggio</h2>

    <?php
    if (isset($_SESSION['message'])) {
        echo '<div class="alert alert-info message">' . $_SESSION['message'] . '</div>';
        unset($_SESSION['message']);
    }
    ?>

    <?php if ($_SESSION['step'] == 1): ?>
    <h4>1. Inserisci un nuovo utente (opzionale)</h4>
    <form method="post" class="row g-3 mb-4">
        <input type="hidden" name="add_user" value="1" />
        <div class="col-md-6">
            <input type="text" name="user_name" class="form-control" placeholder="Nome utente" required>
        </div>
        <div class="col-md-6">
            <input type="email" name="user_email" class="form-control" placeholder="Email utente" required>
        </div>
        <div class="col-12">
            <button class="btn btn-success" type="submit">Aggiungi Utente</button>
        </div>
    </form>

    <h4>2. Seleziona utente e dispositivi da noleggiare</h4>
    <form method="post">
        <input type="hidden" name="step1_submit" value="1" />
        <div class="mb-3">
            <label for="user_id" class="form-label">Seleziona utente</label>
            <select name="user_id" id="user_id" class="form-select" required>
                <option value="">-- Seleziona utente --</option>
                <?php
                $users = $conn->query("SELECT id, name, email FROM users ORDER BY name");
                while ($u = $users->fetch_assoc()) {
                    echo "<option value='{$u['id']}'>" . htmlspecialchars($u['name']) . " ({$u['email']})</option>";
                }
                ?>
            </select>
        </div>
        <div class="mb-3">
            <label for="device_codes" class="form-label">Seleziona dispositivi disponibili (Ctrl/Cmd per multi)</label>
            <select name="device_codes[]" id="device_codes" class="form-select" multiple required size="8" style="height:auto;">
                <?php
                $devs = $conn->query("SELECT codice, numero_inventario, ubicazione FROM device WHERE stato = 'FUNZIONANTE' AND codice NOT IN (SELECT device_code FROM rentals WHERE rental_end IS NULL) ORDER BY ubicazione, codice");
                while ($d = $devs->fetch_assoc()) {
                    echo "<option value='{$d['codice']}'>" . htmlspecialchars($d['codice']) . " - {$d['numero_inventario']} ({$d['ubicazione']})</option>";
                }
                ?>
            </select>
        </div>
        <button type="submit" class="btn btn-primary">Prosegui con la firma</button>
    </form>

    <?php elseif ($_SESSION['step'] == 2): ?>

    <h4>3. Firma l'accordo di noleggio</h4>
    <form method="post" onsubmit="return submitSignature()">
        <input type="hidden" name="step2_submit" value="1" />
        <input type="hidden" name="signature" id="signature" />
        <div>
            <canvas id="signature-pad" width="600" height="200"></canvas>
        </div>
        <div class="signature-container">
            <button type="button" class="btn btn-secondary btn-clear" onclick="clearPad()">Pulisci firma</button>
        </div>
        <br>
        <button type="submit" class="btn btn-success">Completa noleggio e genera PDF</button>
    </form>

    <?php endif; ?>
</div>

<script>
const canvas = document.getElementById('signature-pad');
const ctx = canvas.getContext('2d');
let drawing = false;
let lastPos = {x:0, y:0};

function getMousePos(evt) {
    const rect = canvas.getBoundingClientRect();
    return {x: evt.clientX - rect.left, y: evt.clientY - rect.top};
}
function getTouchPos(evt) {
    const rect = canvas.getBoundingClientRect();
    return {x: evt.touches[0].clientX - rect.left, y: evt.touches[0].clientY - rect.top};
}

canvas.addEventListener('mousedown', (e) => {
    drawing = true;
    lastPos = getMousePos(e);
});
canvas.addEventListener('mouseup', () => { drawing = false; });
canvas.addEventListener('mouseout', () => { drawing = false; });
canvas.addEventListener('mousemove', (e) => {
    if (!drawing) return;
    const mousePos = getMousePos(e);
    ctx.beginPath();
    ctx.moveTo(lastPos.x, lastPos.y);
    ctx.lineTo(mousePos.x, mousePos.y);
    ctx.strokeStyle = 'black';
    ctx.lineWidth = 2;
    ctx.stroke();
    ctx.closePath();
    lastPos = mousePos;
});

// Touch support
canvas.addEventListener('touchstart', (e) => {
    e.preventDefault();
    drawing = true;
    lastPos = getTouchPos(e);
});
canvas.addEventListener('touchend', (e) => {
    e.preventDefault();
    drawing = false;
});
canvas.addEventListener('touchmove', (e) => {
    e.preventDefault();
    if (!drawing) return;
    const touchPos = getTouchPos(e);
    ctx.beginPath();
    ctx.moveTo(lastPos.x, lastPos.y);
    ctx.lineTo(touchPos.x, touchPos.y);
    ctx.strokeStyle = 'black';
    ctx.lineWidth = 2;
    ctx.stroke();
    ctx.closePath();
    lastPos = touchPos;
});

function clearPad() {
    ctx.clearRect(0, 0, canvas.width, canvas.height);
}

function submitSignature() {
    const dataUrl = canvas.toDataURL('image/png');
    if (dataUrl === canvas.toDataURL('image/png').replace(/^data:image\/png;base64,/, '')) {
        alert('Firma obbligatoria!');
        return false;
    }
    document.getElementById('signature').value = dataUrl;
    return true;
}
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
